const express = require('express');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const prisma = require('../config/db');

const router = express.Router();

function signToken(user, roleName) {
  return jwt.sign(
    {
      id: user.id,
      uid: user.uid,
      role: roleName,
      email: user.email,
    },
    process.env.JWT_SECRET,
    { expiresIn: '12h' }
  );
}

// one-time superadmin create
router.post('/bootstrap-superadmin', async (req, res) => {
  try {
    const { name, email, password } = req.body;

    const existing = await prisma.user.findFirst({
      where: { role: { name: 'SUPERADMIN' } },
      include: { role: true },
    });
    if (existing) {
      return res.status(400).json({ message: 'SuperAdmin already exists' });
    }

    // ensure roles exist
    const roleNames = ['DEVELOPER', 'SUPERADMIN', 'AGENT', 'USER'];
    const roles = await Promise.all(
      roleNames.map((name) =>
        prisma.role.upsert({
          where: { name },
          update: {},
          create: { name },
        })
      )
    );
    const superRole = roles.find((r) => r.name === 'SUPERADMIN');

    const passwordHash = await bcrypt.hash(password, 10);

    const superAdmin = await prisma.user.create({
      data: {
        uid: 'SUPERADMIN-1',
        name,
        email,
        passwordHash,
        roleId: superRole.id,
        wallet: { create: { balance: 0 } },
      },
      include: { role: true, wallet: true },
    });

    const token = signToken(superAdmin, superAdmin.role.name);

    return res.status(201).json({
      message: 'SuperAdmin created',
      token,
      user: {
        id: superAdmin.id,
        uid: superAdmin.uid,
        name: superAdmin.name,
        email: superAdmin.email,
        role: superAdmin.role.name,
      },
    });
  } catch (err) {
    console.error('bootstrap-superadmin error:', err);
    return res.status(500).json({ message: 'Internal server error' });
  }
});

// login
router.post('/login', async (req, res) => {
  try {
    const { email, password } = req.body;
    const user = await prisma.user.findUnique({
      where: { email },
      include: { role: true },
    });
    if (!user) return res.status(401).json({ message: 'Invalid credentials' });
    if (user.isBlocked) return res.status(403).json({ message: 'Account is blocked' });

    const ok = await bcrypt.compare(password, user.passwordHash);
    if (!ok) return res.status(401).json({ message: 'Invalid credentials' });

    const token = signToken(user, user.role.name);

    return res.json({
      token,
      user: {
        id: user.id,
        uid: user.uid,
        name: user.name,
        email: user.email,
        role: user.role.name,
      },
    });
  } catch (err) {
    console.error('login error:', err);
    return res.status(500).json({ message: 'Internal server error' });
  }
});

module.exports = router;
