const express = require('express');
const prisma = require('../config/db');
const { authMiddleware, requireRole } = require('../middleware/auth');

const router = express.Router();

// SUPERADMIN creates round manually (for now)
router.post('/rounds', authMiddleware, requireRole('SUPERADMIN'), async (req, res) => {
  try {
    const { openAt, closeAt } = req.body;
    const open = openAt ? new Date(openAt) : new Date();
    const close = closeAt ? new Date(closeAt) : new Date(open.getTime() + 30 * 1000);

    const roundKey = `R-${Date.now()}`;

    const round = await prisma.gameRound.create({
      data: {
        roundKey,
        openAt: open,
        closeAt: close,
        status: 'OPEN',
      },
    });

    return res.status(201).json(round);
  } catch (err) {
    console.error('create round error:', err);
    return res.status(500).json({ message: 'Internal server error' });
  }
});

// list recent rounds (USER/AGENT/SUPERADMIN)
router.get('/rounds', authMiddleware, async (req, res) => {
  try {
    const limit = parseInt(req.query.limit || '10', 10);
    const rounds = await prisma.gameRound.findMany({
      orderBy: { createdAt: 'desc' },
      take: limit,
    });
    return res.json(rounds);
  } catch (err) {
    console.error('list rounds error:', err);
    return res.status(500).json({ message: 'Internal server error' });
  }
});

// USER places bet
router.post('/bet', authMiddleware, requireRole('USER'), async (req, res) => {
  try {
    const userId = req.user.id;
    const { roundId, color, amount } = req.body;
    const amt = parseInt(amount, 10);

    if (!roundId || !color || !amt || amt <= 0) {
      return res.status(400).json({ message: 'Invalid bet data' });
    }

    const round = await prisma.gameRound.findUnique({ where: { id: roundId } });
    if (!round) return res.status(404).json({ message: 'Round not found' });
    if (round.status !== 'OPEN') {
      return res.status(400).json({ message: 'Round is not open for betting' });
    }

    const now = new Date();
    if (now < round.openAt || now > round.closeAt) {
      return res.status(400).json({ message: 'Betting window is closed' });
    }

    const wallet = await prisma.wallet.findUnique({ where: { userId } });
    if (!wallet || wallet.balance < amt) {
      return res.status(400).json({ message: 'Insufficient balance' });
    }

    const result = await prisma.$transaction(async (tx) => {
      const updatedWallet = await tx.wallet.update({
        where: { userId },
        data: { balance: { decrement: amt } },
      });
      const bet = await tx.bet.create({
        data: {
          userId,
          gameRoundId: round.id,
          color,
          amount: amt,
        },
      });
      return { updatedWallet, bet };
    });

    return res.status(201).json({
      message: 'Bet placed successfully',
      walletBalance: result.updatedWallet.balance,
      bet: result.bet,
    });
  } catch (err) {
    console.error('place bet error:', err);
    return res.status(500).json({ message: 'Internal server error' });
  }
});

module.exports = router;
