import React, { useEffect, useState } from 'react';
import api from './api_client';
import UserLayout from './UserLayout';

function formatTime(dateStr) {
  const d = new Date(dateStr);
  return d.toLocaleTimeString();
}

function GamePage({ onLogout }) {
  const [me, setMe] = useState(null);
  const [rounds, setRounds] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [betRoundId, setBetRoundId] = useState('');
  const [betColor, setBetColor] = useState('RED');
  const [betAmount, setBetAmount] = useState('');
  const [betMessage, setBetMessage] = useState('');

  const fetchData = async () => {
    setLoading(true);
    setError('');
    try {
      const [meRes, roundsRes] = await Promise.all([
        api.get('/api/user/me'),
        api.get('/api/game/rounds?limit=10'),
      ]);
      setMe(meRes.data);
      setRounds(roundsRes.data);
      if (roundsRes.data.length > 0 && !betRoundId) {
        setBetRoundId(roundsRes.data[0].id);
      }
    } catch (err) {
      setError(err.response?.data?.message || 'Failed to load game data');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const placeBet = async (e) => {
    e.preventDefault();
    setBetMessage('');
    setError('');
    try {
      const res = await api.post('/api/game/bet', {
        roundId: betRoundId,
        color: betColor,
        amount: parseInt(betAmount, 10),
      });
      setBetMessage(`Bet placed! New balance: ₹ ${res.data.walletBalance}`);
      setBetAmount('');
      await fetchData();
    } catch (err) {
      setError(err.response?.data?.message || 'Failed to place bet');
    }
  };

  const walletBalance = me?.wallet?.balance ?? 0;

  if (!me && loading) return <p>Loading…</p>;

  if (!me && !loading) return <p style={{ color: '#ff4b6e' }}>Could not load user profile.</p>;

  return (
    <UserLayout walletBalance={walletBalance} onLogout={onLogout}>
      {error && <p style={{ color: '#ff4b6e', fontSize: 12, marginBottom: 8 }}>{error}</p>}
      {betMessage && <p style={{ color: '#3cdc7c', fontSize: 12, marginBottom: 8 }}>{betMessage}</p>}
      <section
        style={{
          background: 'rgba(10,10,16,0.9)',
          borderRadius: 20,
          border: '1px solid rgba(246,214,118,0.3)',
          padding: 14,
          marginBottom: 14,
        }}
      >
        <h2 style={{ fontSize: 15, marginBottom: 8, color: 'var(--gold)' }}>Place Your Bet</h2>
        <form
          onSubmit={placeBet}
          style={{
            display: 'grid',
            gridTemplateColumns: '2fr 1.2fr 1.2fr auto',
            gap: 8,
            alignItems: 'center',
          }}
        >
          <select
            style={{ padding: 8, borderRadius: 999, border: '1px solid #444', background: '#050509', color: '#f8f5ff' }}
            value={betRoundId}
            onChange={(e) => setBetRoundId(parseInt(e.target.value, 10))}
          >
            {rounds.map((r) => (
              <option key={r.id} value={r.id}>
                #{r.id} • {r.status} • {formatTime(r.openAt)} - {formatTime(r.closeAt)}
              </option>
            ))}
          </select>
          <select
            style={{ padding: 8, borderRadius: 999, border: '1px solid #444', background: '#050509', color: '#f8f5ff' }}
            value={betColor}
            onChange={(e) => setBetColor(e.target.value)}
          >
            <option value="RED">RED</option>
            <option value="GREEN">GREEN</option>
            <option value="BLUE">BLUE</option>
          </select>
          <input
            style={{ padding: 8, borderRadius: 999, border: '1px solid #444', background: '#050509', color: '#f8f5ff' }}
            type="number"
            min="1"
            placeholder="Amount"
            value={betAmount}
            onChange={(e) => setBetAmount(e.target.value)}
            required
          />
          <button
            type="submit"
            style={{
              padding: 8,
              borderRadius: 999,
              border: 'none',
              background: 'linear-gradient(135deg,#f6d676,#c6a858,#a67c2d)',
              color: '#1a1206',
              fontWeight: 600,
              cursor: 'pointer',
            }}
            disabled={!betRoundId || !betAmount}
          >
            🎰 Bet
          </button>
        </form>
        <p style={{ marginTop: 8, fontSize: 11, color: '#a39ac4' }}>
          Phase 3: manual refresh game. Later we can add countdown, wheel animation and auto-refresh.
        </p>
      </section>

      <section
        style={{
          background: 'rgba(10,10,16,0.9)',
          borderRadius: 20,
          border: '1px solid rgba(246,214,118,0.3)',
          padding: 14,
        }}
      >
        <h2 style={{ fontSize: 15, marginBottom: 8, color: 'var(--gold)' }}>Recent Rounds</h2>
        {rounds.length === 0 && <p style={{ fontSize: 12 }}>No rounds available yet.</p>}
        {rounds.length > 0 && (
          <table style={{ width: '100%', fontSize: 12, borderCollapse: 'collapse' }}>
            <thead>
              <tr style={{ color: '#a39ac4' }}>
                <th style={{ textAlign: 'left', padding: '4px 0' }}>Round</th>
                <th style={{ textAlign: 'left', padding: '4px 0' }}>Status</th>
                <th style={{ textAlign: 'left', padding: '4px 0' }}>Open</th>
                <th style={{ textAlign: 'left', padding: '4px 0' }}>Close</th>
                <th style={{ textAlign: 'left', padding: '4px 0' }}>Result</th>
              </tr>
            </thead>
            <tbody>
              {rounds.map((r) => (
                <tr key={r.id}>
                  <td style={{ padding: '4px 0' }}>{r.roundKey}</td>
                  <td style={{ padding: '4px 0' }}>{r.status}</td>
                  <td style={{ padding: '4px 0' }}>{formatTime(r.openAt)}</td>
                  <td style={{ padding: '4px 0' }}>{formatTime(r.closeAt)}</td>
                  <td style={{ padding: '4px 0' }}>{r.resultColor || 'TBD'}</td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </section>
    </UserLayout>
  );
}

export default GamePage;
